/*----------------------------------------------------------+
| jnrdev #2 example game									|
|															|
| code for the jnrdev tutorial								|
| jnrdev #2 - slopes and a second layer						|
|															|
| read the full article on http://jnrdev.weed-crew.net		|
|															|
|															|
| this sourcecode is released under the GPL.				|
|															|
|															|
| controls:													|
|	cursor keys left/right:	move							|
|	right shift:			jump							|
|	escape:					quit							|
|															|
|															|
| start:		30.12.2003									|
| last changes:	13.01.2004									|
|															|
|						2003-2004  Florian Hufsky (jnrdev)	|
|									      fhufsky@phorus.at	|
|								http://jnrdev.weed-crew.net	|
+----------------------------------------------------------*/


#include "gfx.h"				//gfx: a mini graphics library using SDL




#define MAPWIDTH		32		//width of the map
#define MAPHEIGHT		24		//height of the map
#define TILESIZE		20		//size of the tiles, should be dynamically read

#define PLAYERHEIGHT	37		//should be dynamically read from a player definition file
#define PLAYERWIDTH		11		//--""--



#define WAITTIME		16		//delay between frames


#define	GRAVITATION		1		//gravitation - 1 pixel per frame (too fast, will be changed somewhen (playerx,y -> float, grav: 0.xx))

#define VELMOVING		4		//velocity (speed) for moving left, right
#define VELJUMP			13		//velocity for jumping





class CPlayer{
	public:
		void think();			//handle input, collision detection, ...
		void draw();

		CPlayer();

	private:
		int x, y;			//x, y coordinate (top left of the player rectangle)
		int h, w;			//height, width
		int velx, vely;		//velocity on x, y axis
		
		bool faceright;		//player facing right? -> graphics

		bool lockjump;		//may the player jump
		bool jumping;
		
		int slope_prevtilex;//the tile at slopex, slopey in the last frame
		int slope_prevtiley;



		void collision_detection_map();								//does the whole collision detection

		bool collision_ver(int x, int y, int &tilecoordx);			//tests for collision with a tile on the vertikal line from [x,y] to [x,y+height]
		bool collision_hor_up(int x, int y, int &tilecoordy);		//horizontal line from [x,y] to [x+width, y]
		bool collision_hor_down(int x, int y, int &tilecoordy);		//same as hor_up, but we don't want to go through slopes here
		
		bool collision_slope(int sx, int sy, int &tsx, int &tsy);	//test for collisions against a slope at sx, sy - if a slope is found y is set accordingly

		
		void unlockjump();											//the player may jump again if the conditions in this function are true
};




enum TileType{t_nonsolid, t_solid, t_slopeleft, t_sloperight};

struct CTile{
	TileType	type;	//the type of the tile (solid, slope, ...)
	gfxSprite	*spr;	//sprite to draw

	CTile(){
		type = t_nonsolid;
		spr = NULL;
	};
};


class CMap{
	public:
		void loadMap(const char *file);		//loads the map from a file

		void draw();

		TileType map(int x, int y){			//return the tiletype at the specific position (map coordinates)
			return tiles[x][y].type;
		}

	private:
		CTile tiles[MAPWIDTH][MAPHEIGHT];
};







//spielrelevante variablen
Uint8			*keystates;
SDL_Surface		*screen;		//for gfx

//sprites
gfxSprite		spr_t[7];
gfxSprite		spr_player[2];	//0 left 1 right
gfxSprite		spr_background;


CPlayer			player;
CMap			map;





int main(int argc, char *argv[]){
	unsigned int	framestart;
	SDL_Event		event;

	bool			done;


	//initialize the graphics
	gfx_init(640,480, false);
	//get keystates array
	keystates = SDL_GetKeyState(0);


	spr_t[0].init("gfx/t1.bmp");
	spr_t[1].init("gfx/t2.bmp");
	spr_t[2].init("gfx/t3.bmp", 255, 0, 255);
	spr_t[3].init("gfx/tsloper.bmp", 255, 0, 255);
	spr_t[4].init("gfx/tslopel.bmp", 255, 0, 255);
	spr_t[5].init("gfx/t6.bmp", 255, 0, 255);
	spr_t[6].init("gfx/t7.bmp", 255, 0, 255);
	spr_player[0].init("gfx/left.bmp", 255,0,255);
	spr_player[1].init("gfx/right.bmp", 255,0,255);
	spr_background.init("gfx/bg.bmp");

	
	
	//initialize the map, the player has already been initialized by its constructor
	map.loadMap("maps/map01.map");


	done = false;


	printf("\nentering game loop...\n") ;

	while (!done){
		framestart = SDL_GetTicks();

		//handle messages
		while(SDL_PollEvent(&event)){
			switch(event.type){
				case SDL_QUIT:
					done = true;
				break;

				default:
				break;
			}
		}

		if(keystates[SDLK_ESCAPE])		//quit?
			done = true;



		//update objects
		player.think();


		//draw everything
		spr_background.draw(0,0);
		map.draw();
		player.draw();
		
		

		//printf("possible fps: %.2f\n", 1000.0f / (SDL_GetTicks()-framestart));

		//double buffering -> flip buffers
		SDL_Flip(screen);


		while((SDL_GetTicks()-framestart) < WAITTIME);	//keep framerate constant at 1000/WAITTIME fps
	}

	printf("\n\nshutdown game\n");

	return 0;
}








CPlayer::CPlayer(){
	x		= 100;			//start coordinates
	y		= 100;

	h		= PLAYERHEIGHT;	//save player height and width
	w		= PLAYERWIDTH;

	velx	= 0;			//speed
	vely	= 0;


	lockjump = true;		//player may jump
	faceright = true;		//player looks right
	jumping = false;

	slope_prevtilex = (x + (w>>1)) / TILESIZE;
	slope_prevtiley = (y + h) / TILESIZE;
}




void CPlayer::unlockjump(){		//this function is called if the player hits the ground
	//this if is quite tricky:
	//the player may jump again:
	//a) if he fell of an edge (!jumping) - without releasing the jump key on the ground
	//b) if he jumped - only when he releases the jump key on the ground
	if(!jumping || !keystates[SDLK_RSHIFT]){
		lockjump = false;
		jumping = false;
	}
}




void CPlayer::draw(){
	//crap, change planned in jnrdev #4
	if(faceright)
		spr_player[1].draw(x-4, y-2);
	else
		spr_player[0].draw(x-17, y-2);
}




bool CPlayer::collision_hor_down(int x, int y, int &tilecoordy){
	int tilexpixels = x-(x%TILESIZE);	//calculate the x position (pixels!) of the tiles we check against
	int testend = x + w;		//calculate the end of testing (just to save the x+w calculation each for loop)

	tilecoordy = y/TILESIZE;			//calculate the y position (map coordinates!) of the tiles we want to test

	int tilecoordx = tilexpixels/TILESIZE;	//calculate map x coordinate for first tile


	//loop while the start point (pixels!) of the test tile is inside the players bounding box
	while(tilexpixels <= testend){
		if(map.map(tilecoordx, tilecoordy) == t_solid)	//is a solid tile is found at tilecoordx, tilecoordy?
			return true;	

		tilecoordx++;		//increase tile x map coordinate
		tilexpixels+=TILESIZE;	//increase tile x pixel coordinate
	}

	return false;
}



bool CPlayer::collision_hor_up(int x, int y, int &tilecoordy){
	int tilexpixels = x-(x%TILESIZE);
	int testend = x + w;

	tilecoordy = y/TILESIZE;

	int tilecoordx = tilexpixels/TILESIZE;


	while(tilexpixels <= testend){
		if(map.map(tilecoordx, tilecoordy) != t_nonsolid) //only this changed: when jumping (moving up) we don't want to go through slopes
			return true;	

		tilecoordx++;
		tilexpixels+=TILESIZE;
	}
	return false;
}




//for explanation see CPlayer::collision_hor()
bool CPlayer::collision_ver(int x, int y, int &tilecoordx){
	int tileypixels = y-(y%TILESIZE);
	int testend = y + h;

	tilecoordx = x/TILESIZE;

	int tilecoordy = tileypixels/TILESIZE;

	while(tileypixels <= testend){
		if(map.map(tilecoordx, tilecoordy) == t_solid)
			return true;

		tilecoordy++;
		tileypixels += TILESIZE;
	}

	return false;
}




bool CPlayer::collision_slope(int sx, int sy, int &tsx, int &tsy){	
	tsx = sx / TILESIZE;	//map coordinates of the tile we check against
	tsy = sy / TILESIZE;

	TileType t = map.map(tsx, tsy);
	
	//if we found a slope we set align y to the slope.
	//take a look at jnrdev #2, why it's calculated this way
	if(t == t_sloperight){
		//sloperight -> \  
		y =  (tsy+1)*TILESIZE - (TILESIZE - (sx%TILESIZE))  - h - 1;
		return true;;
	}
	else if(t == t_slopeleft){
		//slopeleft -> /
		y =  (tsy+1)*TILESIZE - sx%TILESIZE  - h - 1 ;
		return true;
	}
	
	return false;
}





void CPlayer::collision_detection_map(){
	//check for slopes (only if moving down)
	if(vely > 0){
		int tsx, tsy;				//slope tile coordinates
		int sx = x + (w>>1) + velx;	//slope chechpoint x coordinate


		if(collision_slope(sx, (y + h), tsx, tsy) ){	//we entered a slope (y is set by collision_slope)
			x += velx;	//move on
			//y has been set by collision_slope
			
			unlockjump();	//we hit the ground - the player may jump again

			vely = 1;		//test against the ground again in the next frame


			slope_prevtilex = tsx;	//save slope koordinate
			slope_prevtiley = tsy;

			return;
		}
		else{				//we're not on a slope this frame - check if we left a slope
			//-1 ... we didn't move from slope to slope
			//0 ... we left a slope after moving down
			//1 ... we left a slope after moving up
			int what = -1;

			if(map.map(slope_prevtilex, slope_prevtiley) == t_sloperight){
				if(velx > 0)	//sloperight + velx > 0  = we left a slope after moving up the slope
					what = 0;
				else
					what = 1;	//we left after moving down the slope
			}
			else if(map.map(slope_prevtilex, slope_prevtiley) == t_slopeleft){
				if(velx < 0)	//sloperight + velx > 0  = we left a slope after moving up the slope
					what = 0;
				else
					what = 1;	//we left after moving down the slope
			}

			if(what != -1){		//if we left a slope and now are on a slope
				int sy;

				if(what == 1){
					y =  tsy*TILESIZE - h -1;		//move y to top of the slope tile
					sy = y + h;
				}
				else{
					y =  (tsy+1)*TILESIZE - h -1;	//move y to the bottom of the slope tile
					sy = y + h + TILESIZE;			//test one tile lower than the bottom of the slope (to test if we move down a long slope)
													//it's physically incorrect, but looks a lot smoother ingame
				}

				
				//check for slopes on new position
				if( collision_slope(sx, sy, tsx, tsy) ){	//slope on new pos (entered a slope after we left a slope)
															//-> we moved from slope to slope
					x += velx;

					unlockjump();

					vely = 1;

					slope_prevtilex = tsx;
					slope_prevtiley = tsy;

					return;
				}
			}	
		}
	}



	//no slope collisions were found -> check for collisions with the map
	int tilecoord;

	//x axis first (--)
	if(velx > 0){		//moving right
		if(collision_ver(x+velx+w, y, tilecoord))	//collision on the right side.
			x = tilecoord*TILESIZE -w-1;			//move to the edge of the tile (tile on the right -> mind the player width)
		else			//no collision
			x += velx;
	}		 
	else if(velx < 0){	//moving left
		if(collision_ver(x+velx, y, tilecoord))		//collision on the left side
			x = (tilecoord+1)*TILESIZE +1;			//move to the edge of the tile
		else
			x += velx;
	}
	

	//then y axis (|)
	if(vely < 0){	//moving up
		if(collision_hor_up(x, y+vely, tilecoord)){
			y		= (tilecoord+1)*TILESIZE +1;
			vely	= 0;
		}
		else{
			y		+= vely;
			vely	+=GRAVITATION;
		}
	}		 
	else{		//moving down / on ground
		//printf("test: down, vely:%d\n", vely);
		if(collision_hor_down(x, y+vely+h, tilecoord)){	//on ground
			y		= tilecoord*TILESIZE -h-1;
			vely	= 1;				//1 so we test against the ground again int the next frame (0 would test against the ground in the next+1 frame)

			unlockjump();
		}
		else{	//falling (in air)
			y		+= vely;
			vely	+= GRAVITATION;

			if(vely >= TILESIZE)		//if the speed is higher than this we might fall through a tile
				vely = TILESIZE;

			lockjump = true;			//don't allow jumping after falling of an edge
		}
	}


	slope_prevtilex = (x + (w>>1)) / TILESIZE;
	slope_prevtiley = (y + h) / TILESIZE;
}






void CPlayer::think(){
	velx=0;	//don't move left / right by default

	if(keystates[SDLK_RIGHT]){
		velx = VELMOVING;		//move right
		faceright = true;		//player graphic is facing right
	}
	if(keystates[SDLK_LEFT]){
		velx = -VELMOVING;		//move left
		faceright = false;		//player graphic is facing left
	}
	if(keystates[SDLK_RSHIFT] && !lockjump){	//if the player isn't jumping already
		vely = -VELJUMP;		//jump!
		lockjump = true;		//player is not allowed to jump anymore
		jumping = true;
	}

	

	collision_detection_map();
}






void CMap::loadMap(const char *file){
	FILE *mapfile;
	int t, i, j;

	mapfile = fopen(file, "rb");

	for(j = 0; j < MAPHEIGHT; j++){
		for(i = 0; i < MAPWIDTH; i++){
			//read solid
			fread(&(tiles[i][j].type), sizeof(TileType), 1, mapfile);
			
			//read tile
			fread(&t, sizeof(int), 1, mapfile);

			if(t != 128)
				tiles[i][j].spr = &spr_t[t];
		}
	}
	fclose(mapfile);
}



void CMap::draw(){
	int i, j;

	for(i = 0; i < MAPWIDTH; i++){
		for(j = 0; j < MAPHEIGHT; j++){
			if(tiles[i][j].spr != NULL)
				tiles[i][j].spr->draw(i*TILESIZE, j*TILESIZE);
		}
	}
}